/*
 * Copyright (C) 2012-2025 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.android.https.imagesearch;

import org.json.JSONException;
import org.json.JSONObject;

import android.os.AsyncTask;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSocketFactory;

public abstract class HttpsImageSearch extends AsyncTask<String, Void, Object> {

    @Override
    protected Object doInBackground(String... params) {
        HttpsURLConnection con1, con2;
        ArrayList<String> imageUrlList = new ArrayList<>();
        byte[] responseArray = null;
        try{
           // --------------------------------------------------------
           // Communication 1st time : Execute image search
           // --------------------------------------------------------
            StringBuilder s = new StringBuilder();
            for (String param : params) {
                s.append(param);
                s.append('+');
            }
            s.deleteCharAt(s.length() - 1);
            // *** POINT 1 *** URI starts with https://.
            // *** POINT 2 *** Sensitive information may be contained in send data.
            // Code for sending image search string is omitted.
            String search_url = "https://www.google.com/search?tbm=isch&q=" +
                    s.toString();

            // *** POINT 3 *** Handle the received data carefully and securely,
            // even though the data was sent from the server connected by HTTPS.
            // Omitted, since this is a sample. Please refer to
            // "3.2 Handling Input Data Carefully and Securely."
            con1 = connectUrl(search_url);
            BufferedReader in = new BufferedReader(
                                    new InputStreamReader(con1.getInputStream()));
           String inputLine;
            StringBuffer sb = new StringBuffer();
            while ((inputLine = in.readLine()) != null) {
               sb.append(inputLine);
            }
            in.close();
            final String regex = "<img.+?src=\"(.+?)\".+?>";
            Pattern pattern = Pattern.compile(regex);
           Matcher matcher = pattern.matcher(sb.toString());
           while (matcher.find()) {
                if (matcher.group(1).startsWith("https://"))
                 imageUrlList.add(matcher.group(1));
            }
           if (imageUrlList == null || imageUrlList.isEmpty()) {
               return null;
            }


            // --------------------------------------------------------
            // Communication 2nd time : Get image
            // --------------------------------------------------------
                
            // *** POINT 1 *** URI starts with https://.
            // *** POINT 2 *** Sensitive information may be contained in send data.
            String image_url = imageUrlList.get(1);
            con2 = connectUrl(image_url);
            checkResponse(con2);
            responseArray = getByteArray(con2);
            if (responseArray == null) {
                return null;
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return responseArray;
    }

    private HttpsURLConnection connectUrl(String strUrl) {
        HttpsURLConnection con = null;
        try {
            SSLContext sc = SSLContext.getInstance("TLS");
            sc.init(null, null, null);
            SSLSocketFactory sf = new NoSSLv3SocketFactory(sc.getSocketFactory());
            HttpsURLConnection.setDefaultSSLSocketFactory(sf);
            URL url = new URL(strUrl);
            con = (HttpsURLConnection) url.openConnection();
            con.setRequestMethod("GET");
            con.connect();
            String cipher_suite = con.getCipherSuite();
            Certificate[] certs = con.getServerCertificates();
        } catch (SSLException e) {
            // *** POINT 4**  Exception handling suitable for the application for
            // SSLException
            e.printStackTrace();// This is sample, so omit the exception process
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();// Exception handling is omitted
        } catch (KeyManagementException e) {
            e.printStackTrace();// Exception handling is omitted
        } catch (ProtocolException e) {
            e.printStackTrace();// Exception handling is omitted
        } catch (MalformedURLException e) {
            e.printStackTrace();// Exception handling is omitted
        } catch (IOException e) {
            e.printStackTrace();// Exception handling is omitted
        }
        return con;
    }

    private byte[] getByteArray(HttpsURLConnection con) {
        byte[] buff = new byte[1024];
        byte[] result = null;
        BufferedInputStream inputStream = null;
        ByteArrayOutputStream responseArray = null;
        int length;

        try {
            inputStream = new BufferedInputStream(con.getInputStream());
            responseArray = new ByteArrayOutputStream();

            while ((length = inputStream.read(buff)) != -1) {
                if (length > 0) {
                    responseArray.write(buff, 0, length);
                }
            }
            result = responseArray.toByteArray();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    // Exception handling is omitted
                }
            }
            if (responseArray != null) {
                try {
                    responseArray.close();
                } catch (IOException e) {
                    // Exception handling is omitted
                }
            }
        }
        return result;
    }

    private void checkResponse(HttpURLConnection response) throws IOException {
        int statusCode = response.getResponseCode();
        if (HttpURLConnection.HTTP_OK != statusCode) {
            throw new IOException("HttpStatus: " + statusCode);
        }
    }
}
